import { z } from 'zod';

const optionalString = z
  .string()
  .trim()
  .max(200)
  .transform((value) => value || null)
  .nullable()
  .optional();

export const productSchema = z.object({
  name: z.string().trim().min(1, 'El nombre es obligatorio').max(200),
  sku: optionalString,
  barcode: optionalString,
  supplier: optionalString,
  currentStock: z.coerce.number().int().min(0).default(0),
  minStock: z.coerce.number().int().min(0).default(0),
  purchaseMultiple: z.coerce.number().int().min(1).default(1),
  isActive: z.boolean().optional().default(true)
});

export const productUpdateSchema = productSchema.partial().extend({
  name: z.string().trim().min(1).max(200).optional()
});

export const movementSchema = z.object({
  type: z.enum(['SALE', 'RESTOCK', 'ADJUSTMENT']),
  productId: z.string().uuid('Producto inválido'),
  quantity: z.coerce.number().int().positive('Cantidad debe ser mayor a 0'),
  note: z.string().trim().max(500).optional().nullable(),
  allowNegative: z.boolean().optional().default(false),
  adminPassword: z.string().trim().max(200).optional()
});

export const importModeSchema = z.enum(['MERGE', 'REPLACE_STOCK', 'ADD_STOCK']);

export const googleSheetsImportSchema = z.object({
  url: z.string().trim().url('URL inválida'),
  gid: z.string().trim().max(30).optional().nullable(),
  mode: importModeSchema
});

export type ImportMode = z.infer<typeof importModeSchema>;
export type ProductInput = z.infer<typeof productSchema>;
export type MovementInput = z.infer<typeof movementSchema>;
