import { MovementType } from '@prisma/client';
import { prisma } from '@/lib/prisma';

function toDelta(type: MovementType, quantity: number) {
  if (type === 'SALE') {
    return -Math.abs(quantity);
  }
  if (type === 'RESTOCK') {
    return Math.abs(quantity);
  }
  return quantity;
}

export async function applyMovement(params: {
  productId: string;
  type: MovementType;
  quantity: number;
  note?: string | null;
  allowNegative?: boolean;
}) {
  const { productId, type, quantity, note, allowNegative = false } = params;

  return prisma.$transaction(async (tx) => {
    const product = await tx.product.findUnique({ where: { id: productId } });

    if (!product || !product.isActive) {
      throw new Error('Producto no encontrado o inactivo');
    }

    const delta = toDelta(type, quantity);
    const newStock = product.currentStock + delta;

    if (newStock < 0 && !allowNegative) {
      throw new Error('Stock insuficiente. Activa Permitir negativo (admin).');
    }

    const updatedProduct = await tx.product.update({
      where: { id: product.id },
      data: { currentStock: newStock }
    });

    const movement = await tx.movement.create({
      data: {
        type,
        productId: product.id,
        quantity: delta,
        note: note?.trim() || null
      }
    });

    return { updatedProduct, movement };
  });
}
