import { MovementType, PrismaClient } from '@prisma/client';
import Papa from 'papaparse';
import { ImportMode } from '@/lib/validation';

type RawRecord = Record<string, unknown>;

type NormalizedRow = {
  rowNumber: number;
  name?: string;
  sku?: string;
  barcode?: string;
  supplier?: string;
  stock?: number;
  minStock?: number;
  purchaseMultiple?: number;
};

type RowResult = {
  rowNumber: number;
  status: 'created' | 'updated' | 'failed';
  productName?: string;
  reason?: string;
};

export type ImportSummary = {
  created: number;
  updated: number;
  failed: number;
  rows: RowResult[];
};

const keyMap: Record<string, keyof Omit<NormalizedRow, 'rowNumber'>> = {
  name: 'name',
  nombre: 'name',
  sku: 'sku',
  codigo: 'sku',
  barcode: 'barcode',
  codigo_barras: 'barcode',
  supplier: 'supplier',
  proveedor: 'supplier',
  currentstock: 'stock',
  stock: 'stock',
  stock_actual: 'stock',
  minstock: 'minStock',
  minimo: 'minStock',
  stock_minimo: 'minStock',
  purchasemultiple: 'purchaseMultiple',
  multiplo_compra: 'purchaseMultiple'
};

function normalizeHeader(header: string) {
  return header.toLowerCase().trim().replace(/\s+/g, '_');
}

function cleanText(value: unknown) {
  if (value === undefined || value === null) {
    return undefined;
  }
  const text = String(value).trim();
  if (!text) {
    return undefined;
  }
  return text.slice(0, 200);
}

function cleanNumber(value: unknown) {
  if (value === undefined || value === null || value === '') {
    return undefined;
  }
  const parsed = Number(String(value).replace(',', '.'));
  if (!Number.isFinite(parsed)) {
    return undefined;
  }
  return Math.max(0, Math.floor(parsed));
}

export function parseCsvText(text: string): RawRecord[] {
  const parsed = Papa.parse<RawRecord>(text, {
    header: true,
    skipEmptyLines: true,
    transformHeader: (header) => normalizeHeader(header)
  });

  if (parsed.errors.length) {
    throw new Error(parsed.errors[0]?.message || 'CSV inválido');
  }

  return parsed.data;
}

export function normalizeRows(records: RawRecord[]): NormalizedRow[] {
  return records.map((record, index) => {
    const row: NormalizedRow = { rowNumber: index + 2 };

    for (const [rawKey, rawValue] of Object.entries(record)) {
      const mapped = keyMap[normalizeHeader(rawKey)];
      if (!mapped) {
        continue;
      }

      if (mapped === 'stock' || mapped === 'minStock' || mapped === 'purchaseMultiple') {
        (row as any)[mapped] = cleanNumber(rawValue);
      } else {
        (row as any)[mapped] = cleanText(rawValue);
      }
    }

    if (row.purchaseMultiple !== undefined) {
      row.purchaseMultiple = Math.max(1, row.purchaseMultiple);
    }

    return row;
  });
}

function getLookup(row: NormalizedRow) {
  if (row.sku) {
    return { field: 'sku' as const, value: row.sku };
  }
  if (row.barcode) {
    return { field: 'barcode' as const, value: row.barcode };
  }
  if (row.name) {
    return { field: 'name' as const, value: row.name };
  }
  return null;
}

export async function processImport(prisma: PrismaClient, rows: NormalizedRow[], mode: ImportMode): Promise<ImportSummary> {
  const summary: ImportSummary = {
    created: 0,
    updated: 0,
    failed: 0,
    rows: []
  };

  for (const row of rows) {
    const lookup = getLookup(row);

    if (!lookup) {
      summary.failed += 1;
      summary.rows.push({
        rowNumber: row.rowNumber,
        status: 'failed',
        reason: 'Debe incluir sku, barcode o nombre'
      });
      continue;
    }

    try {
      await prisma.$transaction(async (tx) => {
        let existing = await tx.product.findFirst({
          where: {
            isActive: true,
            [lookup.field]: lookup.value
          }
        });

        const data: Record<string, unknown> = {};

        if (row.name !== undefined) data.name = row.name;
        if (row.sku !== undefined) data.sku = row.sku;
        if (row.barcode !== undefined) data.barcode = row.barcode;
        if (row.supplier !== undefined) data.supplier = row.supplier;
        if (row.minStock !== undefined) data.minStock = row.minStock;
        if (row.purchaseMultiple !== undefined) data.purchaseMultiple = Math.max(1, row.purchaseMultiple);

        if (!existing) {
          if (!row.name) {
            throw new Error('Nombre es obligatorio para crear producto nuevo');
          }

          const initialStock = row.stock ?? 0;
          existing = await tx.product.create({
            data: {
              name: row.name,
              sku: row.sku,
              barcode: row.barcode,
              supplier: row.supplier,
              minStock: row.minStock ?? 0,
              purchaseMultiple: Math.max(1, row.purchaseMultiple ?? 1),
              currentStock: initialStock,
              isActive: true
            }
          });

          await tx.movement.create({
            data: {
              type: MovementType.IMPORT,
              productId: existing.id,
              quantity: initialStock,
              note: `Importación (${mode}) - alta inicial`
            }
          });

          summary.created += 1;
          summary.rows.push({
            rowNumber: row.rowNumber,
            status: 'created',
            productName: existing.name
          });
          return;
        }

        let newStock = existing.currentStock;
        if (row.stock !== undefined) {
          if (mode === 'ADD_STOCK') {
            newStock = existing.currentStock + row.stock;
          } else {
            newStock = row.stock;
          }
        }

        const updated = await tx.product.update({
          where: { id: existing.id },
          data: {
            ...data,
            ...(row.stock !== undefined ? { currentStock: newStock } : {})
          }
        });

        const stockDelta = newStock - existing.currentStock;
        await tx.movement.create({
          data: {
            type: MovementType.IMPORT,
            productId: existing.id,
            quantity: stockDelta,
            note: stockDelta === 0 ? `Importación (${mode}) - sin cambio de stock` : `Importación (${mode})`
          }
        });

        summary.updated += 1;
        summary.rows.push({
          rowNumber: row.rowNumber,
          status: 'updated',
          productName: updated.name
        });
      });
    } catch (error) {
      summary.failed += 1;
      summary.rows.push({
        rowNumber: row.rowNumber,
        status: 'failed',
        reason: error instanceof Error ? error.message : 'Error desconocido'
      });
    }
  }

  return summary;
}

export function buildGoogleSheetsCsvUrl(inputUrl: string, gid?: string | null) {
  const url = inputUrl.trim();

  // Publicado como CSV directo
  if (url.includes('output=csv') || url.endsWith('.csv')) {
    return url;
  }

  const match = url.match(/\/spreadsheets\/d\/([a-zA-Z0-9-_]+)/);
  if (!match) {
    throw new Error('URL de Google Sheets inválida. Usa un enlace de hoja de cálculo válido.');
  }

  const sheetId = match[1];
  const extractedGid = gid || new URL(url).searchParams.get('gid');
  const gidParam = extractedGid ? `&gid=${encodeURIComponent(extractedGid)}` : '';

  return `https://docs.google.com/spreadsheets/d/${sheetId}/export?format=csv${gidParam}`;
}
