'use client';

import { useEffect, useMemo, useState } from 'react';
import { useToast } from '@/components/toast';
import { SkeletonList } from '@/components/skeleton-list';

type RestockItem = {
  id: string;
  name: string;
  sku: string | null;
  supplier: string | null;
  currentStock: number;
  minStock: number;
  purchaseMultiple: number;
  suggestedOrder: number;
};

export function RestockList() {
  const { showToast } = useToast();
  const [items, setItems] = useState<RestockItem[]>([]);
  const [loading, setLoading] = useState(true);
  const [supplier, setSupplier] = useState('');

  async function loadData(selectedSupplier = supplier) {
    setLoading(true);
    try {
      const response = await fetch(`/api/restock${selectedSupplier ? `?supplier=${encodeURIComponent(selectedSupplier)}` : ''}`);
      if (!response.ok) throw new Error('No se pudo cargar reabastecimiento');
      const data = (await response.json()) as RestockItem[];
      setItems(data);
    } catch {
      showToast('Error al cargar reabastecimiento', 'error');
    } finally {
      setLoading(false);
    }
  }

  useEffect(() => {
    void loadData('');
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, []);

  const suppliers = useMemo(() => {
    const values = Array.from(new Set(items.map((item) => item.supplier).filter(Boolean))) as string[];
    return values.sort((a, b) => a.localeCompare(b));
  }, [items]);

  return (
    <div className="mx-auto max-w-3xl px-4 pb-24 pt-4">
      <div className="flex items-center justify-between gap-2">
        <h1 className="text-2xl font-bold text-slate-900">Reabastecimiento</h1>
        <a
          href={`/api/restock.csv${supplier ? `?supplier=${encodeURIComponent(supplier)}` : ''}`}
          className="rounded-xl border border-slate-300 px-3 py-2 text-sm font-semibold"
        >
          Exportar CSV
        </a>
      </div>

      <div className="mt-3 flex gap-2">
        <select
          value={supplier}
          onChange={(event) => {
            const value = event.target.value;
            setSupplier(value);
            void loadData(value);
          }}
          className="flex-1 rounded-xl border border-slate-300 bg-white px-3 py-3"
        >
          <option value="">Todos los proveedores</option>
          {suppliers.map((value) => (
            <option key={value} value={value}>
              {value}
            </option>
          ))}
        </select>
        <button onClick={() => window.print()} className="rounded-xl bg-brand-600 px-4 py-3 text-white">
          Imprimir
        </button>
      </div>

      <div className="mt-4 space-y-3 print:mt-2">
        {loading ? (
          <SkeletonList rows={6} />
        ) : (
          items.map((item) => (
            <div key={item.id} className="rounded-2xl border border-slate-200 bg-white p-4 print:break-inside-avoid">
              <p className="text-lg font-semibold">{item.name}</p>
              <p className="text-sm text-slate-600">Proveedor: {item.supplier || '-'}</p>
              <p className="text-sm text-slate-600">
                Stock: {item.currentStock} / Mínimo: {item.minStock} / Múltiplo: {item.purchaseMultiple}
              </p>
              <p className="mt-2 text-base font-bold text-brand-700">Sugerido: {item.suggestedOrder}</p>
            </div>
          ))
        )}

        {!loading && !items.length && (
          <p className="rounded-xl bg-white p-4 text-sm text-slate-500">Sin productos por reabastecer.</p>
        )}
      </div>
    </div>
  );
}
