'use client';

import { FormEvent, useEffect, useMemo, useState } from 'react';
import { useToast } from '@/components/toast';
import { SkeletonList } from '@/components/skeleton-list';

type Product = {
  id: string;
  name: string;
  sku: string | null;
  barcode: string | null;
  currentStock: number;
  minStock: number;
  supplier: string | null;
};

type QuickMovementProps = {
  type: 'SALE' | 'RESTOCK';
  title: string;
  subtitle: string;
  confirmLabel: string;
};

export function QuickMovement({ type, title, subtitle, confirmLabel }: QuickMovementProps) {
  const { showToast } = useToast();
  const [query, setQuery] = useState('');
  const [products, setProducts] = useState<Product[]>([]);
  const [loading, setLoading] = useState(false);
  const [selected, setSelected] = useState<Product | null>(null);
  const [qty, setQty] = useState(1);
  const [submitting, setSubmitting] = useState(false);
  const [allowNegative, setAllowNegative] = useState(false);
  const [adminModalOpen, setAdminModalOpen] = useState(false);
  const [adminPassword, setAdminPassword] = useState('');

  useEffect(() => {
    const controller = new AbortController();
    const timeout = setTimeout(async () => {
      setLoading(true);
      try {
        const response = await fetch(`/api/products?query=${encodeURIComponent(query)}`, {
          signal: controller.signal
        });

        if (!response.ok) {
          throw new Error('No se pudieron cargar productos');
        }

        const data = (await response.json()) as Product[];
        setProducts(data);
      } catch {
        if (!controller.signal.aborted) {
          showToast('Error al buscar productos', 'error');
        }
      } finally {
        if (!controller.signal.aborted) {
          setLoading(false);
        }
      }
    }, 220);

    return () => {
      clearTimeout(timeout);
      controller.abort();
    };
  }, [query, showToast]);

  const projectedStock = useMemo(() => {
    if (!selected) return null;
    if (type === 'SALE') {
      return selected.currentStock - qty;
    }
    return selected.currentStock + qty;
  }, [selected, qty, type]);

  async function submitMovement(event: FormEvent) {
    event.preventDefault();
    if (!selected) return;

    if (!navigator.onLine) {
      showToast('Sin conexión: las acciones de escritura requieren internet.', 'error');
      return;
    }

    if (type === 'SALE' && (projectedStock ?? 0) < 0 && !allowNegative) {
      showToast('El stock quedaría negativo. Activa Permitir negativo (admin).', 'error');
      return;
    }

    setSubmitting(true);
    try {
      const response = await fetch('/api/movements', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
          type,
          productId: selected.id,
          quantity: qty,
          allowNegative,
          adminPassword: allowNegative ? adminPassword : undefined
        })
      });

      const data = await response.json();
      if (!response.ok) {
        throw new Error(data.error || 'No se pudo guardar el movimiento');
      }

      showToast('Movimiento registrado', 'success');
      setSelected((prev) => (prev ? { ...prev, currentStock: data.updatedProduct.currentStock } : null));
      setQty(1);
      setAllowNegative(false);
      setAdminPassword('');
      setAdminModalOpen(false);

      // refresca lista
      const listResponse = await fetch(`/api/products?query=${encodeURIComponent(query)}`);
      if (listResponse.ok) {
        const list = (await listResponse.json()) as Product[];
        setProducts(list);
      }
    } catch (error) {
      showToast(error instanceof Error ? error.message : 'Error al registrar movimiento', 'error');
    } finally {
      setSubmitting(false);
    }
  }

  return (
    <div className="mx-auto flex max-w-3xl flex-col gap-4 pb-24">
      <header className="px-4 pt-4">
        <h1 className="text-2xl font-bold text-slate-900">{title}</h1>
        <p className="mt-1 text-sm text-slate-600">{subtitle}</p>
      </header>

      <div className="sticky top-0 z-10 bg-slate-100 px-4 pb-3 pt-2">
        <input
          value={query}
          onChange={(event) => setQuery(event.target.value)}
          placeholder="Buscar por nombre, SKU o código"
          className="w-full rounded-2xl border border-slate-300 bg-white px-4 py-4 text-lg shadow-sm focus:border-brand-500 focus:outline-none"
        />
      </div>

      <div className="px-4">
        {loading ? (
          <SkeletonList rows={5} />
        ) : (
          <div className="space-y-3">
            {products.map((product) => (
              <button
                key={product.id}
                type="button"
                onClick={() => {
                  setSelected(product);
                  setQty(1);
                  setAllowNegative(false);
                  setAdminPassword('');
                }}
                className={`w-full rounded-2xl border p-4 text-left shadow-sm transition ${
                  selected?.id === product.id
                    ? 'border-brand-600 bg-brand-50'
                    : 'border-slate-200 bg-white active:scale-[0.99]'
                }`}
              >
                <p className="text-lg font-semibold text-slate-900">{product.name}</p>
                <p className="text-sm text-slate-600">SKU: {product.sku || '-'} | Stock: {product.currentStock}</p>
              </button>
            ))}
            {!products.length && <p className="rounded-xl bg-white p-4 text-sm text-slate-500">Sin resultados.</p>}
          </div>
        )}
      </div>

      {selected && (
        <form onSubmit={submitMovement} className="mx-4 rounded-2xl border border-slate-200 bg-white p-4 shadow-sm">
          <p className="text-base font-semibold text-slate-900">{selected.name}</p>
          <p className="text-sm text-slate-600">Stock actual: {selected.currentStock}</p>

          <div className="mt-4 flex items-center justify-between rounded-xl border border-slate-200 p-2">
            <button
              type="button"
              className="h-12 w-12 rounded-xl bg-slate-100 text-2xl"
              onClick={() => setQty((value) => Math.max(1, value - 1))}
            >
              -
            </button>
            <span className="text-2xl font-bold">{qty}</span>
            <button
              type="button"
              className="h-12 w-12 rounded-xl bg-slate-100 text-2xl"
              onClick={() => setQty((value) => value + 1)}
            >
              +
            </button>
          </div>

          <p className="mt-3 text-sm text-slate-600">Stock proyectado: {projectedStock}</p>

          {type === 'SALE' && (projectedStock ?? 0) < 0 && (
            <div className="mt-3 rounded-xl border border-amber-300 bg-amber-50 p-3 text-sm">
              <label className="flex items-center gap-2 font-medium text-amber-800">
                <input
                  type="checkbox"
                  checked={allowNegative}
                  onChange={(event) => {
                    const checked = event.target.checked;
                    if (checked) {
                      setAdminModalOpen(true);
                    } else {
                      setAllowNegative(false);
                      setAdminPassword('');
                    }
                  }}
                />
                Permitir negativo (admin)
              </label>
            </div>
          )}

          <button
            disabled={submitting}
            className="mt-4 w-full rounded-2xl bg-brand-600 px-4 py-4 text-lg font-semibold text-white disabled:opacity-60"
            type="submit"
          >
            {submitting ? 'Guardando...' : confirmLabel}
          </button>
        </form>
      )}

      {adminModalOpen && (
        <div className="fixed inset-0 z-50 flex items-center justify-center bg-black/40 p-4">
          <div className="w-full max-w-sm rounded-2xl bg-white p-4">
            <h3 className="text-lg font-semibold">Confirmar password admin</h3>
            <p className="mt-1 text-sm text-slate-600">Necesario para permitir stock negativo en venta.</p>
            <input
              type="password"
              value={adminPassword}
              onChange={(event) => setAdminPassword(event.target.value)}
              className="mt-4 w-full rounded-xl border border-slate-300 px-3 py-3"
              placeholder="Password admin"
            />
            <div className="mt-4 flex gap-2">
              <button
                type="button"
                className="flex-1 rounded-xl border border-slate-300 px-3 py-2"
                onClick={() => {
                  setAdminModalOpen(false);
                  setAllowNegative(false);
                  setAdminPassword('');
                }}
              >
                Cancelar
              </button>
              <button
                type="button"
                className="flex-1 rounded-xl bg-brand-600 px-3 py-2 text-white"
                onClick={() => {
                  if (!adminPassword.trim()) {
                    showToast('Ingresa el password admin', 'error');
                    return;
                  }
                  setAllowNegative(true);
                  setAdminModalOpen(false);
                }}
              >
                Confirmar
              </button>
            </div>
          </div>
        </div>
      )}
    </div>
  );
}
