'use client';

import { useEffect, useMemo, useState } from 'react';
import { z } from 'zod';
import { useToast } from '@/components/toast';

const formSchema = z.object({
  name: z.string().trim().min(1, 'Nombre obligatorio'),
  sku: z.string().trim().optional(),
  barcode: z.string().trim().optional(),
  supplier: z.string().trim().optional(),
  currentStock: z.coerce.number().int().min(0),
  minStock: z.coerce.number().int().min(0),
  purchaseMultiple: z.coerce.number().int().min(1)
});

type Product = {
  id: string;
  name: string;
  sku: string | null;
  barcode: string | null;
  supplier: string | null;
  currentStock: number;
  minStock: number;
  purchaseMultiple: number;
};

type FormState = {
  name: string;
  sku: string;
  barcode: string;
  supplier: string;
  currentStock: number;
  minStock: number;
  purchaseMultiple: number;
};

const emptyForm: FormState = {
  name: '',
  sku: '',
  barcode: '',
  supplier: '',
  currentStock: 0,
  minStock: 0,
  purchaseMultiple: 1
};

export function ProductEditor() {
  const { showToast } = useToast();
  const [products, setProducts] = useState<Product[]>([]);
  const [query, setQuery] = useState('');
  const [loading, setLoading] = useState(true);
  const [saving, setSaving] = useState(false);
  const [editing, setEditing] = useState<Product | null>(null);
  const [form, setForm] = useState<FormState>(emptyForm);
  const [modalOpen, setModalOpen] = useState(false);

  async function loadProducts(search = '') {
    setLoading(true);
    try {
      const response = await fetch(`/api/products?query=${encodeURIComponent(search)}`);
      if (!response.ok) throw new Error('No se pudo cargar productos');
      const data = (await response.json()) as Product[];
      setProducts(data);
    } catch {
      showToast('Error al cargar productos', 'error');
    } finally {
      setLoading(false);
    }
  }

  useEffect(() => {
    void loadProducts('');
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, []);

  useEffect(() => {
    const timeout = setTimeout(() => {
      void loadProducts(query);
    }, 250);
    return () => clearTimeout(timeout);
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [query]);

  const modalTitle = useMemo(() => (editing ? 'Editar producto' : 'Nuevo producto'), [editing]);

  function openCreate() {
    setEditing(null);
    setForm(emptyForm);
    setModalOpen(true);
  }

  function openEdit(product: Product) {
    setEditing(product);
    setForm({
      name: product.name,
      sku: product.sku || '',
      barcode: product.barcode || '',
      supplier: product.supplier || '',
      currentStock: product.currentStock,
      minStock: product.minStock,
      purchaseMultiple: product.purchaseMultiple
    });
    setModalOpen(true);
  }

  async function saveProduct() {
    if (!navigator.onLine) {
      showToast('Sin conexión: no se pueden guardar cambios.', 'error');
      return;
    }

    const parsed = formSchema.safeParse(form);
    if (!parsed.success) {
      showToast(parsed.error.issues[0]?.message || 'Formulario inválido', 'error');
      return;
    }

    setSaving(true);
    try {
      const payload = {
        ...parsed.data,
        sku: parsed.data.sku || null,
        barcode: parsed.data.barcode || null,
        supplier: parsed.data.supplier || null
      };

      const response = await fetch(editing ? `/api/products/${editing.id}` : '/api/products', {
        method: editing ? 'PUT' : 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(payload)
      });

      const data = await response.json();
      if (!response.ok) throw new Error(data.error || 'No se pudo guardar');

      showToast(editing ? 'Producto actualizado' : 'Producto creado', 'success');
      setEditing(null);
      setForm(emptyForm);
      setModalOpen(false);
      await loadProducts(query);
    } catch (error) {
      showToast(error instanceof Error ? error.message : 'Error al guardar', 'error');
    } finally {
      setSaving(false);
    }
  }

  async function softDeleteProduct(productId: string) {
    if (!navigator.onLine) {
      showToast('Sin conexión: no se pueden eliminar productos.', 'error');
      return;
    }

    if (!confirm('¿Desactivar producto?')) return;

    try {
      const response = await fetch(`/api/products/${productId}`, { method: 'DELETE' });
      const data = await response.json();
      if (!response.ok) throw new Error(data.error || 'No se pudo eliminar');

      showToast('Producto desactivado', 'success');
      await loadProducts(query);
    } catch (error) {
      showToast(error instanceof Error ? error.message : 'Error al desactivar', 'error');
    }
  }

  return (
    <div className="mx-auto max-w-3xl px-4 pb-24 pt-4">
      <div className="flex items-center justify-between gap-2">
        <h1 className="text-2xl font-bold text-slate-900">Productos</h1>
        <button onClick={openCreate} className="rounded-xl bg-brand-600 px-4 py-3 text-sm font-semibold text-white">
          + Agregar
        </button>
      </div>

      <input
        value={query}
        onChange={(event) => setQuery(event.target.value)}
        placeholder="Buscar producto"
        className="mt-4 w-full rounded-2xl border border-slate-300 bg-white px-4 py-3"
      />

      <div className="mt-4 space-y-3">
        {loading && <p className="rounded-xl bg-white p-4 text-sm text-slate-500">Cargando...</p>}
        {!loading &&
          products.map((product) => (
            <div key={product.id} className="rounded-2xl border border-slate-200 bg-white p-4">
              <p className="text-lg font-semibold">{product.name}</p>
              <p className="text-sm text-slate-600">Stock: {product.currentStock} | Min: {product.minStock}</p>
              <p className="text-sm text-slate-600">Proveedor: {product.supplier || '-'}</p>
              <div className="mt-3 flex gap-2">
                <button
                  onClick={() => openEdit(product)}
                  className="flex-1 rounded-xl border border-slate-300 px-3 py-2 text-sm font-semibold"
                >
                  Editar
                </button>
                <button
                  onClick={() => void softDeleteProduct(product.id)}
                  className="flex-1 rounded-xl bg-red-600 px-3 py-2 text-sm font-semibold text-white"
                >
                  Desactivar
                </button>
              </div>
            </div>
          ))}

        {!loading && !products.length && (
          <p className="rounded-xl bg-white p-4 text-sm text-slate-500">No hay productos.</p>
        )}
      </div>

      {modalOpen && (
        <div className="fixed inset-0 z-50 bg-black/40 p-4">
          <div className="mx-auto max-w-md rounded-2xl bg-white p-4">
            <h2 className="text-lg font-semibold">{modalTitle}</h2>
            <div className="mt-3 grid gap-3">
              <input
                value={form.name}
                onChange={(event) => setForm((prev) => ({ ...prev, name: event.target.value }))}
                placeholder="Nombre"
                className="rounded-xl border border-slate-300 px-3 py-2"
              />
              <input
                value={form.sku}
                onChange={(event) => setForm((prev) => ({ ...prev, sku: event.target.value }))}
                placeholder="SKU"
                className="rounded-xl border border-slate-300 px-3 py-2"
              />
              <input
                value={form.barcode}
                onChange={(event) => setForm((prev) => ({ ...prev, barcode: event.target.value }))}
                placeholder="Código de barras"
                className="rounded-xl border border-slate-300 px-3 py-2"
              />
              <input
                value={form.supplier}
                onChange={(event) => setForm((prev) => ({ ...prev, supplier: event.target.value }))}
                placeholder="Proveedor"
                className="rounded-xl border border-slate-300 px-3 py-2"
              />
              <input
                type="number"
                value={form.currentStock}
                onChange={(event) => setForm((prev) => ({ ...prev, currentStock: Number(event.target.value) }))}
                placeholder="Stock actual"
                className="rounded-xl border border-slate-300 px-3 py-2"
              />
              <input
                type="number"
                value={form.minStock}
                onChange={(event) => setForm((prev) => ({ ...prev, minStock: Number(event.target.value) }))}
                placeholder="Stock mínimo"
                className="rounded-xl border border-slate-300 px-3 py-2"
              />
              <input
                type="number"
                value={form.purchaseMultiple}
                onChange={(event) => setForm((prev) => ({ ...prev, purchaseMultiple: Number(event.target.value) }))}
                placeholder="Múltiplo de compra"
                className="rounded-xl border border-slate-300 px-3 py-2"
              />
            </div>
            <div className="mt-4 flex gap-2">
              <button
                onClick={() => {
                  setEditing(null);
                  setForm(emptyForm);
                  setModalOpen(false);
                }}
                className="flex-1 rounded-xl border border-slate-300 px-3 py-2"
              >
                Cancelar
              </button>
              <button
                onClick={() => void saveProduct()}
                disabled={saving}
                className="flex-1 rounded-xl bg-brand-600 px-3 py-2 text-white disabled:opacity-60"
              >
                {saving ? 'Guardando...' : 'Guardar'}
              </button>
            </div>
          </div>
        </div>
      )}
    </div>
  );
}
