'use client';

import { FormEvent, useEffect, useState } from 'react';
import { useToast } from '@/components/toast';

type ImportMode = 'MERGE' | 'REPLACE_STOCK' | 'ADD_STOCK';

type ImportRow = {
  rowNumber: number;
  status: 'created' | 'updated' | 'failed';
  productName?: string;
  reason?: string;
};

type ImportSummary = {
  created: number;
  updated: number;
  failed: number;
  rows: ImportRow[];
  sourceUrl?: string;
};

const MODES: { value: ImportMode; label: string }[] = [
  { value: 'MERGE', label: 'Merge (actualiza campos enviados)' },
  { value: 'REPLACE_STOCK', label: 'Replace stock (fija stock exacto)' },
  { value: 'ADD_STOCK', label: 'Add stock (suma al stock actual)' }
];

export function ImportManager() {
  const { showToast } = useToast();
  const [mode, setMode] = useState<ImportMode>('MERGE');
  const [file, setFile] = useState<File | null>(null);
  const [loadingFile, setLoadingFile] = useState(false);
  const [loadingSheets, setLoadingSheets] = useState(false);
  const [summary, setSummary] = useState<ImportSummary | null>(null);
  const [sheetUrl, setSheetUrl] = useState('');
  const [sheetGid, setSheetGid] = useState('');

  useEffect(() => {
    const storedUrl = localStorage.getItem('inventario:sheets:url') || '';
    const storedGid = localStorage.getItem('inventario:sheets:gid') || '';
    setSheetUrl(storedUrl);
    setSheetGid(storedGid);
  }, []);

  async function submitFile(event: FormEvent) {
    event.preventDefault();
    if (!file) {
      showToast('Selecciona un archivo', 'error');
      return;
    }

    if (!navigator.onLine) {
      showToast('Sin conexión: la importación requiere internet.', 'error');
      return;
    }

    setLoadingFile(true);
    try {
      const form = new FormData();
      form.set('file', file);
      form.set('mode', mode);

      const response = await fetch('/api/import/file', {
        method: 'POST',
        body: form
      });
      const data = (await response.json()) as ImportSummary & { error?: string };
      if (!response.ok) throw new Error(data.error || 'No se pudo importar archivo');

      setSummary(data);
      showToast('Importación completada', 'success');
    } catch (error) {
      showToast(error instanceof Error ? error.message : 'Error en importación', 'error');
    } finally {
      setLoadingFile(false);
    }
  }

  async function submitGoogleSheets(event: FormEvent) {
    event.preventDefault();

    if (!sheetUrl.trim()) {
      showToast('Ingresa la URL de Google Sheets', 'error');
      return;
    }

    if (!navigator.onLine) {
      showToast('Sin conexión: la sincronización requiere internet.', 'error');
      return;
    }

    setLoadingSheets(true);
    try {
      const response = await fetch('/api/import/google-sheets', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
          url: sheetUrl,
          gid: sheetGid || undefined,
          mode
        })
      });

      const data = (await response.json()) as ImportSummary & { error?: string };
      if (!response.ok) throw new Error(data.error || 'No se pudo sincronizar');

      setSummary(data);
      showToast('Sincronización completada', 'success');
    } catch (error) {
      showToast(error instanceof Error ? error.message : 'Error al sincronizar', 'error');
    } finally {
      setLoadingSheets(false);
    }
  }

  return (
    <div className="mx-auto max-w-3xl px-4 pb-24 pt-4">
      <h1 className="text-2xl font-bold text-slate-900">Importar productos</h1>
      <p className="mt-1 text-sm text-slate-600">Carga Excel/CSV o sincroniza desde Google Sheets (sin OAuth).</p>

      <label className="mt-4 block text-sm font-semibold text-slate-700">Modo de importación</label>
      <select
        value={mode}
        onChange={(event) => setMode(event.target.value as ImportMode)}
        className="mt-1 w-full rounded-xl border border-slate-300 bg-white px-3 py-3"
      >
        {MODES.map((item) => (
          <option key={item.value} value={item.value}>
            {item.label}
          </option>
        ))}
      </select>

      <form onSubmit={submitFile} className="mt-4 rounded-2xl border border-slate-200 bg-white p-4">
        <h2 className="text-lg font-semibold">A) Subir archivo Excel o CSV</h2>
        <input
          type="file"
          accept=".xlsx,.csv"
          onChange={(event) => setFile(event.target.files?.[0] || null)}
          className="mt-3 w-full rounded-xl border border-slate-300 px-3 py-2"
        />
        <button
          type="submit"
          disabled={loadingFile}
          className="mt-4 w-full rounded-xl bg-brand-600 px-4 py-3 font-semibold text-white disabled:opacity-60"
        >
          {loadingFile ? 'Importando...' : 'Importar archivo'}
        </button>
      </form>

      <form onSubmit={submitGoogleSheets} className="mt-4 rounded-2xl border border-slate-200 bg-white p-4">
        <h2 className="text-lg font-semibold">B) Sync desde Google Sheets</h2>
        <p className="mt-1 text-sm text-slate-600">
          Usa una URL CSV publicada o pega una URL normal + GID. Si falla, publica la hoja en la web como CSV.
        </p>
        <input
          value={sheetUrl}
          onChange={(event) => setSheetUrl(event.target.value)}
          placeholder="URL de Google Sheets"
          className="mt-3 w-full rounded-xl border border-slate-300 px-3 py-2"
        />
        <input
          value={sheetGid}
          onChange={(event) => setSheetGid(event.target.value)}
          placeholder="GID (opcional)"
          className="mt-2 w-full rounded-xl border border-slate-300 px-3 py-2"
        />
        <button
          type="submit"
          disabled={loadingSheets}
          className="mt-4 w-full rounded-xl bg-brand-600 px-4 py-3 font-semibold text-white disabled:opacity-60"
        >
          {loadingSheets ? 'Sincronizando...' : 'Sincronizar Google Sheets'}
        </button>
      </form>

      {summary && (
        <div className="mt-4 rounded-2xl border border-slate-200 bg-white p-4">
          <h3 className="text-lg font-semibold">Resultado de importación</h3>
          <p className="mt-1 text-sm text-slate-700">
            Creados: {summary.created} | Actualizados: {summary.updated} | Fallidos: {summary.failed}
          </p>
          {summary.sourceUrl && (
            <p className="mt-1 break-all text-xs text-slate-500">Fuente: {summary.sourceUrl}</p>
          )}

          <div className="mt-3 max-h-72 overflow-auto rounded-xl border border-slate-200">
            <table className="w-full text-left text-sm">
              <thead className="sticky top-0 bg-slate-50">
                <tr>
                  <th className="px-3 py-2">Fila</th>
                  <th className="px-3 py-2">Estado</th>
                  <th className="px-3 py-2">Detalle</th>
                </tr>
              </thead>
              <tbody>
                {summary.rows.map((row) => (
                  <tr key={`${row.rowNumber}-${row.status}`} className="border-t border-slate-100">
                    <td className="px-3 py-2">{row.rowNumber}</td>
                    <td className="px-3 py-2">
                      <span
                        className={`rounded-full px-2 py-1 text-xs font-semibold ${
                          row.status === 'failed'
                            ? 'bg-red-100 text-red-700'
                            : row.status === 'created'
                              ? 'bg-emerald-100 text-emerald-700'
                              : 'bg-blue-100 text-blue-700'
                        }`}
                      >
                        {row.status}
                      </span>
                    </td>
                    <td className="px-3 py-2">{row.reason || row.productName || '-'}</td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
        </div>
      )}
    </div>
  );
}
