import { NextRequest, NextResponse } from 'next/server';
import { requireApiAuth } from '@/lib/auth';
import { prisma } from '@/lib/prisma';
import { calculateSuggestedOrder, sanitizeCsvCell } from '@/lib/restock';

export async function GET(request: NextRequest) {
  const authError = await requireApiAuth(request);
  if (authError) return authError;

  const supplier = request.nextUrl.searchParams.get('supplier')?.trim();

  const products = await prisma.product.findMany({
    where: {
      isActive: true,
      ...(supplier ? { supplier } : {})
    },
    orderBy: [{ supplier: 'asc' }, { name: 'asc' }]
  });

  const headers = ['Nombre', 'SKU', 'Código barras', 'Proveedor', 'Stock actual', 'Stock mínimo', 'Múltiplo', 'Sugerido'];

  const lines = [headers.join(',')];

  for (const product of products.filter((item) => item.currentStock <= item.minStock)) {
    const row = [
      sanitizeCsvCell(product.name),
      sanitizeCsvCell(product.sku || ''),
      sanitizeCsvCell(product.barcode || ''),
      sanitizeCsvCell(product.supplier || ''),
      product.currentStock,
      product.minStock,
      product.purchaseMultiple,
      calculateSuggestedOrder(product)
    ].map((value) => `"${String(value).replaceAll('"', '""')}"`);

    lines.push(row.join(','));
  }

  const csv = lines.join('\n');

  return new NextResponse(csv, {
    status: 200,
    headers: {
      'Content-Type': 'text/csv; charset=utf-8',
      'Content-Disposition': 'attachment; filename="reabastecimiento.csv"'
    }
  });
}
