import { NextRequest, NextResponse } from 'next/server';
import * as XLSX from 'xlsx';
import { requireApiAuth } from '@/lib/auth';
import { prisma } from '@/lib/prisma';
import { importModeSchema } from '@/lib/validation';
import { normalizeRows, parseCsvText, processImport } from '@/lib/import-utils';

const MAX_ROWS = 5000;

export async function POST(request: NextRequest) {
  const authError = await requireApiAuth(request);
  if (authError) return authError;

  const form = await request.formData();
  const modeRaw = form.get('mode');
  const modeParsed = importModeSchema.safeParse(modeRaw);

  if (!modeParsed.success) {
    return NextResponse.json({ error: 'Modo de importación inválido' }, { status: 400 });
  }

  const uploaded = form.get('file');
  if (!(uploaded instanceof File)) {
    return NextResponse.json({ error: 'Debes subir un archivo CSV o XLSX' }, { status: 400 });
  }

  const fileName = uploaded.name.toLowerCase();

  try {
    const buffer = Buffer.from(await uploaded.arrayBuffer());
    let records: Record<string, unknown>[] = [];

    if (fileName.endsWith('.csv')) {
      const text = buffer.toString('utf-8');
      records = parseCsvText(text);
    } else if (fileName.endsWith('.xlsx')) {
      const workbook = XLSX.read(buffer, { type: 'buffer' });
      const firstSheet = workbook.SheetNames[0];
      if (!firstSheet) {
        return NextResponse.json({ error: 'El archivo XLSX no tiene hojas' }, { status: 400 });
      }

      records = XLSX.utils.sheet_to_json(workbook.Sheets[firstSheet], {
        defval: '',
        raw: false
      }) as Record<string, unknown>[];
    } else {
      return NextResponse.json({ error: 'Formato no soportado. Usa .csv o .xlsx' }, { status: 400 });
    }

    if (!records.length) {
      return NextResponse.json({ error: 'El archivo está vacío' }, { status: 400 });
    }

    if (records.length > MAX_ROWS) {
      return NextResponse.json(
        { error: `El archivo supera el máximo de ${MAX_ROWS} filas por importación` },
        { status: 400 }
      );
    }

    const rows = normalizeRows(records);
    const summary = await processImport(prisma, rows, modeParsed.data);

    return NextResponse.json(summary);
  } catch (error) {
    return NextResponse.json(
      {
        error: error instanceof Error ? error.message : 'No se pudo procesar el archivo'
      },
      { status: 400 }
    );
  }
}
