import { NextRequest, NextResponse } from 'next/server';
import { z } from 'zod';
import { applySessionCookie, createSessionToken, getAdminPassword } from '@/lib/auth';
import { checkRateLimit } from '@/lib/rate-limit';

const loginSchema = z.object({
  password: z.string().min(1, 'Password requerido')
});

function getClientIp(request: NextRequest) {
  return (
    request.headers.get('x-forwarded-for')?.split(',')[0]?.trim() ||
    request.headers.get('x-real-ip') ||
    'unknown'
  );
}

export async function POST(request: NextRequest) {
  const ip = getClientIp(request);
  const limit = checkRateLimit(`login:${ip}`);

  if (!limit.allowed) {
    return NextResponse.json(
      { error: `Demasiados intentos. Intenta en ${limit.retryAfter} segundos.` },
      { status: 429 }
    );
  }

  let body: unknown;
  try {
    body = await request.json();
  } catch {
    return NextResponse.json({ error: 'JSON inválido' }, { status: 400 });
  }

  const parsed = loginSchema.safeParse(body);
  if (!parsed.success) {
    return NextResponse.json({ error: parsed.error.issues[0]?.message || 'Datos inválidos' }, { status: 400 });
  }

  const validPassword = getAdminPassword();
  if (parsed.data.password !== validPassword) {
    return NextResponse.json({ error: 'Password incorrecto' }, { status: 401 });
  }

  const token = await createSessionToken();
  const response = NextResponse.json({ ok: true });
  return applySessionCookie(response, token);
}
